/*global define, require */
/*jslint white: true */

/*
	Envelope (Attack, Delay, Sustain, Release):

	This file implements various envelope filters.
*/

define ([	"src/math/mathUtils",	"src/math/TimeGraph",		"src/utils"],
function(	mathUtils,				TimeGraph,					utils ) {
	'use strict';

	// Each timegraph should start at zero, relative to the previous.
	return function Envelope(inAttack, inDelay, inSustain, inRelease) {
		var that = this;

		that.attack		= inAttack ? inAttack.clone() : undefined;
		that.delay		= inDelay ? inDelay.clone() : undefined;
		that.sustain	= inSustain ? inSustain.clone() : undefined;
		that.release	= inRelease ? inRelease.clone() : undefined;

		that.transition	= new TimeGraph([[[0, 0], [0.1, 0.5], [0.5, 0.9], [1, 1]]]);
		that.transitionDuration = 0.05;

		that.timeScale = undefined;	// optionally defined.  Scales time to lengthen or shorten ADR response
		that.yScale = undefined;
		that.signalOn = undefined;
		that.envelopeT = undefined;

		that.evalStack = [];

		function addEvalStack(inEnvelope, inT, inSignalOn, inOffsetT, inTransitionGraph, inTransitionDuration, inYScale) {
			var	addEval = {onT: inSignalOn ? inT : undefined, offT: inSignalOn ? undefined : inT,
							offsetT : inOffsetT || 0, signalOn: inSignalOn, transitionGraph: inTransitionGraph,
							transitionDuration: inTransitionDuration, yScale: inYScale};
			inEnvelope.evalStack.push(addEval);
		}

		function throwBadEnvelopeError() {
			throw new Error("Incorrect time graph formulation.");
		}

		that.setTimeScale = function (inTimeScale) {
			this.timeScale = inTimeScale;
		};

		that.getTimeScale = function () {
			return this.timeScale;
		};

		function convertToEnvelopeTime(inEnvelope, inT) {
			var tScale = inEnvelope.getTimeScale() || 1;
			return tScale * inT;
		}

		function getEvalStackItem(inEnvelope, inEvalStackIndex) {
			var	stackSize = inEnvelope.evalStack.length, result;

			if (stackSize > 0) {
				result = inEnvelope.evalStack[inEvalStackIndex];
			}

			return result;
		}

		function getAttackDuration(inEnvelope) {
			var	result = 0;

			if (inEnvelope.attack) {
				result = inEnvelope.attack.getXMax();
			}
			return result;
		}

		function getDelayDuration(inEnvelope) {
			var	result = 0;

			if (inEnvelope.delay) {
				result += inEnvelope.delay.getXMax();
			}
			return result;
		}

		function getReleaseDuration(inEnvelope) {
			var	result = 0;

			if (inEnvelope.release) {
				result += inEnvelope.release.getXMax();
			}
			return result;
		}

		function getSustainLoopDuration(inEnvelope) {
			var	result = 0;

			if (inEnvelope.sustain) {
				result += inEnvelope.sustain.getXMax();
			}
			return result;
		}

		function getTransitionDuration(inEvalStackItem) {
			var	result = 0;

			if (inEvalStackItem.transitionGraph && inEvalStackItem.transitionDuration) {
				result += inEvalStackItem.transitionDuration;
			}
			return result;
		}

		function evaluateStackItem(inEnvelope, inEvalStackItem, inT) {
			var	result = {value : undefined, inactive : false}, previousEndTime, dTime, duration, evalTime = 0;

			if (inEvalStackItem.onT !== undefined && inEvalStackItem.offT === undefined) {
				// The item started from Attack and hasn't released.
				dTime = inT - (inEvalStackItem.onT + inEvalStackItem.offsetT);

				if (inEnvelope.attack && getAttackDuration(inEnvelope) > dTime) {
					result.value = inEnvelope.attack.evaluate(dTime);
				} else if (inEnvelope.delay && ((getAttackDuration(inEnvelope) + getDelayDuration(inEnvelope)) > dTime)) {
					result.value = inEnvelope.delay.evaluate(dTime - getAttackDuration(inEnvelope));
				} else {
					if (inEnvelope.sustain !== undefined) {
						duration = getSustainLoopDuration(inEnvelope);
						if (duration) {
							dTime -= getAttackDuration(inEnvelope) + getDelayDuration(inEnvelope);
							evalTime = mathUtils.loop(dTime / duration, "cycle") * duration;
						}
						result.value = inEnvelope.sustain.evaluate(evalTime);
					} else if (inEnvelope.delay) {
						result.value = inEnvelope.delay.evaluate(inEnvelope.delay.getXMax());
					} else if (inEnvelope.attack) {
						result.value = inEnvelope.attack.evaluate(inEnvelope.attack.getXMax());
					} else {
						result.value = 1;
					}
				}
			} else if (inEvalStackItem.offT !== undefined) {
				// The item is within the Release cycle.
				dTime = inT - (inEvalStackItem.offT + inEvalStackItem.offsetT);
				if (inEnvelope.release && getReleaseDuration(inEnvelope) > dTime) {
					result.value = inEnvelope.release.evaluate(dTime);
				} else {
					result.inactive = true;
				}
			}

			if (result.value && inEvalStackItem.yScale !== undefined) {
				result.value *= inEvalStackItem.yScale;
			}

			return result;
		}

		function evaluateStack(inEnvelope, inT, inEvalStackIndex) {
			var evalItem, resultValue, itemResult, transitionWeight = 1, transitionT, dT, blendValue;

			inEvalStackIndex = (inEvalStackIndex === undefined) ? inEnvelope.evalStack.length - 1 : inEvalStackIndex;

			evalItem = getEvalStackItem(inEnvelope, inEvalStackIndex);

			if (evalItem) {
				itemResult = evaluateStackItem(inEnvelope, evalItem, inT);
				resultValue = itemResult.value;
				if (evalItem.transitionGraph) {
					transitionT = (evalItem.onT || evalItem.offT);	// not offset
					dT = inT - transitionT;	// not scaled...
					if (dT >= 0 && evalItem.transitionDuration && dT < evalItem.transitionDuration) {
						transitionWeight = dT / evalItem.transitionDuration;
						transitionWeight = evalItem.transitionGraph.evaluate(transitionWeight);
						resultValue = resultValue || 0;
						resultValue = transitionWeight * resultValue;

						transitionWeight = 1 - transitionWeight;
						if (transitionWeight) {
							blendValue = evaluateStack(inEnvelope, inT, inEvalStackIndex - 1);
							if (blendValue) {
								resultValue += transitionWeight * blendValue;
							}
							itemResult.inactive = false;	// This item isn't inactive until the transition graph is no longer used.
						}
					} else {
						evalItem.transitionGraph = undefined;	// No longer relavent.
					}
					if (itemResult.inactive) {
						inEnvelope.evalStack[inEvalStackIndex] = undefined;	// Mark for removal.
					}
				}
			}

			return resultValue;
		}

		function updateEvalStackDueToSignalChange(inEnvelope, inT, inSignalOn) {
			var	lastItem, yScale, dTime, offsetT, duration, transitionDuration = inEnvelope.transitionDuration,
				currentValue, previousValue;

			lastItem = getEvalStackItem(inEnvelope, inEnvelope.evalStack.length - 1);

			// If the signal state changes, append a state change transition onto the stack
			if (lastItem && lastItem.signalOn !== inSignalOn) {
				if (lastItem.signalOn) {
					// Was on, now off.
					dTime = inT - (lastItem.onT + lastItem.offsetT);
					duration = getAttackDuration(inEnvelope);
					if (duration > 0 && dTime < duration) {
						offsetT = -getReleaseDuration(inEnvelope) * (1 - dTime / duration);
						transitionDuration = (getReleaseDuration(inEnvelope) + offsetT);
						transitionDuration *= 0.5;
					}
					// Determine yMax;
					previousValue = evaluateStack(inEnvelope, inT);
				} else {
					// Was off, now on.
					dTime = inT - (lastItem.offT + lastItem.offsetT);
					duration = getReleaseDuration(inEnvelope);
					if (duration > 0 && dTime < duration) {
						offsetT = -getAttackDuration(inEnvelope) * (1 - dTime / duration);
						transitionDuration = (getAttackDuration(inEnvelope) + offsetT);
						transitionDuration *= 0.5;
					}
				}
				// Add something new to the eval stack
				addEvalStack(inEnvelope, inT, inSignalOn, offsetT, inEnvelope.transition, transitionDuration, yScale);
				if (previousValue !== undefined) {
					lastItem = getEvalStackItem(inEnvelope, inEnvelope.evalStack.length - 1);
					currentValue = evaluateStackItem(inEnvelope, lastItem, inT);
					if (currentValue && currentValue.value) {
						lastItem.yScale = previousValue / currentValue.value;
					}
				}
			}
		}

		function removeInvalidItems(inEnvelope) {
			var	newItems = [], i, inactiveItemFound;

			for (i = inEnvelope.evalStack.length - 1; i >= 0; i -= 1) {
				if (inactiveItemFound === undefined || inactiveItemFound !== true) {
					inactiveItemFound = inEnvelope.evalStack[i] === undefined;
				}
				if (inactiveItemFound) {
					inEnvelope.evalStack[i] = undefined;
					// All previous items must be invalid too.
				}
			}

			for (i = 0; i < inEnvelope.evalStack.length; i += 1) {
				if (inEnvelope.evalStack[i]) {
					newItems.push(inEnvelope.evalStack[i]);
				} else {
					// All previous items must be invalid too.  Break to disregard them.
					break;
				}
			}
			inEnvelope.evalStack = newItems;
		}

		that.evaluate = function (inT, inSignalOn) {
			var	stackSize = this.evalStack.length, lastItem, result;

			inT = convertToEnvelopeTime(this, inT);
			this.signalOn = inSignalOn;
			if (stackSize > 0) {
				lastItem = getEvalStackItem(this, stackSize - 1);
			}
			if (lastItem === undefined && inSignalOn === true) {
				// Add something to the eval stack since we are starting the envelope.
				addEvalStack(this, inT, true);
			}

			// If the signal state changes, append a state change transition onto the stack
			if (lastItem && lastItem.signalOn !== inSignalOn) {
				updateEvalStackDueToSignalChange(this, inT, inSignalOn);
			}

			result = evaluateStack(this, inT);

			// Clear older transitions
			removeInvalidItems(this);

			return result;
		};

		that.clone = function () {
			var newTimeGraph = utils.clone(true, {}, this);
			return newTimeGraph;
		};

		return that;
	};
});
